// $License: NOLICENSE
//--------------------------------------------------------------------------------
/**
  Функции по взаимодействию с сервером ТехноДок

  @file $relPath
  @copyright $copyright
  @author SMS-Automation
*/

//--------------------------------------------------------------------------------
// Libraries used (#uses)

#uses "Technodoc/Common/technodocNet"
#uses "Technodoc/Core/technodocServerSettings"
#uses "Technodoc/Core/technodocAuthorization"

//--------------------------------------------------------------------------------
// Variables and Constants

//--------------------------------------------------------------------------------
// Константы для проверки доступности ТехноДока (используется для проверки доступности ТехноДока в скрипте мониторинга Technodoc.Server.ctl и панели technodoc.pnl)
// Веб-метод для проверки состояния сервера
const string TECHNODOC_CHECK_HEALTH_WEB_METHOD_NAME = "/health";

// Результат веб-метода healthWebMethodName, при котором считается, что сервер в порядке
const string TECHNODOC_HEALTHY_SERVER_STATUS_NAME = "Healthy";

// Флаг отладки выполнения
const string TECHNODOC_ALIVE_STATUS_DEBUG_FLAG = "TechnodocAliveStatus";

// Веб-метод для проверки состояния сервера
const string TECHNODOC_SERVER_STATUS_WEB_METHOD_NAME = "/api/ClusterServer/GetStatus";

// Главный сервер имеет значение 2
const string PRIMARY_SERVER_STATUS = "2";

// Веб-метод для получения версии сервера
const string TECHNODOC_SERVER_VERSION_METHOD_NAME = "/api/CoreConfiguration/GetVersion";

//--------------------------------------------------------------------------------
//@public members
//--------------------------------------------------------------------------------

/** Доступен ли сервер ТехноДока
  @param baseTechnodocUrl Базовый адрес ТехноДока
  @return Возвращает true если сервер доступен, иначе false
*/
bool isTechnodocServerAlive(string baseTechnodocUrl)
{
  mapping result;
  string healthCheckUrl = baseTechnodocUrl + TECHNODOC_CHECK_HEALTH_WEB_METHOD_NAME;
  technodocNetGet(healthCheckUrl, result);
  DebugFTN(TECHNODOC_ALIVE_STATUS_DEBUG_FLAG, "Результат выполнения проверки доступности по адресу \"" + healthCheckUrl + "\": " + result[TECHNODOC_STATUS_CODE_RESPONSE_HEADER_NAME]);
  return result[TECHNODOC_STATUS_CODE_RESPONSE_HEADER_NAME] == TECHNODOC_HEALTHY_SERVER_STATUS_NAME;
}

/** Является ли указанный сервер ТехноДока основным
  @param baseTechnodocUrl Адрес основного сервера ТехноДока
  @return Возвращает true если сервер основной, иначе false
*/
bool isTechnodocServerPrimary(string baseTechnodocUrl)
{
  mapping result;
  string getServerStatusUrl = baseTechnodocUrl + TECHNODOC_SERVER_STATUS_WEB_METHOD_NAME;

  technodocNetGet(getServerStatusUrl, result);
  DebugFTN(TECHNODOC_ALIVE_STATUS_DEBUG_FLAG, "Результат выполнения проверки статуса по адресу \"" + getServerStatusUrl + "\": " + result[TECHNODOC_CONTENT_RESPONSE_HEADER_NAME]);
  return result[TECHNODOC_CONTENT_RESPONSE_HEADER_NAME] == PRIMARY_SERVER_STATUS;
}

/** Вернуть текущий основной сервер ТехноДока
  @param primaryServerAddress Адрес основного сервера ТехноДока
  @return Возвращает true если сервер найден, иначе false
*/
bool getPrimaryTechnodocAddress(string& primaryServerAddress)
{
  dyn_string primaryServerAddresses;
  int primaryServerAddressIndex;

  for(int i = 1; i <= dynlen(TECHNODOC_CLUSTER_URLS); i++)
  {
    if(isTechnodocServerPrimary(TECHNODOC_CLUSTER_URLS[i]))
    {
      primaryServerAddressIndex = i;
      dynAppend(primaryServerAddresses, TECHNODOC_CLUSTER_URLS[i]);
      break;
    }
  }

  int countPrimaryServerAddresses = dynlen(primaryServerAddresses);
  if(countPrimaryServerAddresses != 1)
  {
    string message = countPrimaryServerAddresses == 0
      ? "Не найдено ни одного основного сервера."
      : "Найдено более одного основного сервера.";
    DebugTN(message);
    return false;
  }

  if (primaryServerAddresses[1] != TECHNODOC_CLUSTER_URLS[1]) {
    dynRemove(TECHNODOC_CLUSTER_URLS, primaryServerAddressIndex);
    dynInsertAt(TECHNODOC_CLUSTER_URLS, primaryServerAddresses[1], 1);
  }

  primaryServerAddress = primaryServerAddresses[1];
  return true;
}

/** Вернуть базовый адрес текущего сервера на основе текущего url в webview
  @param currentUrl Адрес текущей страницы
  @param urls Список адресов кластера
  @return Базовый url текущего работающего сервера
*/
string getCurrentTechnodocBaseUrl(string currentUrl)
{
  string currentBaseUrl = "";
  for(int i = 1; i <= dynlen(TECHNODOC_CLUSTER_URLS); i++)
  {
    if(strpos(currentUrl, TECHNODOC_CLUSTER_URLS[i]) >= 0)
    {
      currentBaseUrl = TECHNODOC_CLUSTER_URLS[i];
      break;
    }
  }

  return currentBaseUrl;
}


/** Вернуть адрес активного сервера ТехноДок в кластере
  @return Адрес активного сервера ТехноДок в кластере
*/
string getActiveServerUrl() {
  for (int i = 1; i < dynlen(TECHNODOC_CLUSTER_URLS) + 1; i++)
  {
    if(isTechnodocServerPrimary(TECHNODOC_CLUSTER_URLS[i])) {
      return TECHNODOC_CLUSTER_URLS[i];
    }
  }
  return "";
}


/** Вернуть версию приложения сервера
  @param baseTechnodocUrl Адрес сервера
  @param version Версия приложения
  @return Возвращает true если версию сервера удалось получить иначе false
*/
bool getVersion(string baseTechnodocUrl, string& version)
{
  mapping result;
  string getServerStatusUrl = baseTechnodocUrl + TECHNODOC_SERVER_VERSION_METHOD_NAME;

  int returnCode = technodocNetGet(getServerStatusUrl, result);

  if(returnCode != 0)
  {
    DebugTN("Не удалось определить версию севрера");
    return false;
  }

  version = result[TECHNODOC_CONTENT_RESPONSE_HEADER_NAME];

  return true;
}

//--------------------------------------------------------------------------------
//@private members
//--------------------------------------------------------------------------------

