var RolesSecurityCtrl = (function () {
    function RolesSecurityCtrl($scope, $location, $modal, notificationService, language, roleProvider, permissionProvider, authentication, permissionService) {
        var _this = this;
        this.$scope = $scope;
        this.$location = $location;
        this.$modal = $modal;
        this.notificationService = notificationService;
        this.language = language;
        this.roleProvider = roleProvider;
        this.permissionProvider = permissionProvider;
        this.authentication = authentication;
        this.permissionService = permissionService;
        this.tree = new SmsTree();
        this.plugins = ['types', 'wholerow', 'grid', 'checkbox', 'search'];
        this.types = new SmsTypes('glyphicon glyphicon-none', 'fa fa-folder', 'fa fa-file-o');
        this.expandLevel = 0;
        this.checkbox = new SmsTreeCheckBox(true, true, true, false, 'undetermined', true, false, false);
        this.nodes = [];
        this.isAddNewRoleModalOpened = false;
        this.permissions = [];
        this.originalRole = { id: undefined, name: undefined, isSystem: undefined, permissions: [] };
        this.watchFunctions = [];
        this.columns = [new SmsTreeColumn('', 'Text')];
        this.startingData = new SmsTreeDataItem('1', this.language.getElementValue('loadingHdr'));
        this.watchFunctions.push(this.$scope.$on('jstreeReady', function (event, treeId) {
            if (_this.tree.element.attr('id') === treeId && !_this.isAddNewRoleModalOpened) {
                _this.loadTreePermissions();
            }
        }));
        this.watchFunctions.push(this.$scope.$on('jstreeRefresh', function () {
            _this.loadRoles();
        }));
        this.watchFunctions.push(this.$scope.$on('jstreeChanged', function (event, data) {
            if (_.isUndefined(_this.selectedRole) || _this.isAddNewRoleModalOpened) {
                return;
            }
            if (data.action === 'deselect_node' || data.action === 'select_node') {
                _this.selectedRole.permissions = _this.permissions
                    .filter(function (permission) { return data.selected.indexOf(permission.ident) !== -1; })
                    .map(function (permission) {
                    return {
                        ident: permission.ident,
                        name: permission.name
                    };
                });
            }
        }));
        this.watchFunctions.push(this.$scope.$on('$locationChangeStart', function (event) {
            var newUrl = _this.$location.url();
            if ((_.isUndefined($scope.lastLocation) || newUrl !== $scope.lastLocation) &&
                _this.isSelectedRoleModified()) {
                _this.showSaveChangesModal(function () { return _this.$location.url(newUrl); });
                event.preventDefault();
            }
            $scope.lastLocation = _this.isSelectedRoleModified() ? undefined : newUrl;
        }));
    }
    RolesSecurityCtrl.prototype.$onDestroy = function () {
        _.forEach(this.watchFunctions, function (watchFunction) { return watchFunction(); });
    };
    RolesSecurityCtrl.prototype.loadTreePermissions = function () {
        var _this = this;
        this.permissionProvider
            .getPermissionGroups()
            .then(function (permissionGroupTree) {
            _this.permissions = _this.getPermissions(permissionGroupTree);
            _this.createTree(permissionGroupTree, '#');
            _this.tree.element.jstree().settings.core.data = _this.nodes;
            _this.tree.element.jstree().refresh(false, true);
        })
            .catch(function (data) {
            _this.notificationService.errorMessage(_this.language.getElementValue('failedLoadDataServerMsg'), data);
        });
    };
    RolesSecurityCtrl.prototype.getPermissions = function (tree) {
        var _this = this;
        return _.flatten(tree.map(function (group) {
            return group.childrens.length === 0 ? group.permissions : _this.getPermissions(group.childrens);
        }));
    };
    RolesSecurityCtrl.prototype.createTree = function (tree, parent) {
        var _this = this;
        tree.forEach(function (item) {
            if (item.childrens.length !== 0) {
                _this.nodes.push(new SmsTreeDataItem(item.ident, item.name, 'folder', parent));
                _this.createTree(item.childrens, item.ident);
            }
            else {
                _this.nodes.push(new SmsTreeDataItem(item.ident, item.name, 'folder', parent));
                item.permissions.forEach(function (permission) {
                    var nodePermission = new SmsTreeDataItem(permission.ident, permission.name, 'file', item.ident, new SmsTreeDataItemAttr(permission.name));
                    _this.nodes.push(nodePermission);
                });
            }
        });
    };
    RolesSecurityCtrl.prototype.loadRoles = function (newSelectedRole) {
        var _this = this;
        if (newSelectedRole === void 0) { newSelectedRole = undefined; }
        this.roleProvider.getAll().then(function (allRoles) {
            var _a;
            _this.roles = _(allRoles)
                .chain()
                .sortBy(function (role) { return role.name; })
                .sortBy(function (role) { return !role.isSystem; })
                .value();
            _this.select((_a = _.find(_this.roles, function (role) { return role.id === (newSelectedRole === null || newSelectedRole === void 0 ? void 0 : newSelectedRole.id); })) !== null && _a !== void 0 ? _a : _this.roles[0]);
        });
    };
    RolesSecurityCtrl.prototype.showSaveChangesModal = function (action) {
        var _this = this;
        if (!this.isSelectedRoleModified()) {
            action();
            return;
        }
        this.$modal
            .open({
            templateUrl: 'ng1/dist/src/Common/Components/YesNoCancelModal.html',
            controller: 'yesNoCancelModalCtrl',
            windowClass: 'app-modal-window',
            backdrop: 'static',
            resolve: {
                header: function () {
                    return _this.language.getElementValue('securityHdr');
                },
                body: function () {
                    return String.Format(_this.language.getElementValue('securityRoleSaveChangesMsg'), _this.selectedRole.name);
                }
            }
        })
            .result.then(function (data) {
            if (data === 'ok') {
                _this.update();
                action();
            }
            if (data === 'no') {
                _this.reset(_this.selectedRole);
                action();
            }
        }, function () {
            return;
        });
    };
    RolesSecurityCtrl.prototype.select = function (role) {
        var _a, _b;
        this.selectedRole = role;
        this.originalRole.name = role.name;
        this.originalRole.permissions = (_a = role.permissions) !== null && _a !== void 0 ? _a : [];
        if (this.selectedRole.permissions == null) {
            this.selectedRole.permissions = [];
        }
        else {
            this.tree.element.jstree().deselect_all(true);
            var rolePermissionsIdents = _.map(this.selectedRole.permissions, function (permission) { return permission.ident; });
            this.tree.element.jstree().select_node(rolePermissionsIdents, true, true);
            this.tree.element.jstree()._undetermined();
            this.tree.element.jstree().enable_node(this.nodes);
            if ((_b = this.selectedRole) === null || _b === void 0 ? void 0 : _b.isSystem) {
                this.tree.element.jstree().disable_node(this.permissionService.permissions.Administrate);
            }
        }
    };
    RolesSecurityCtrl.prototype.reset = function (role) {
        if (_.isUndefined(role)) {
            return;
        }
        role.name = this.originalRole.name;
        role.permissions = this.originalRole.permissions;
    };
    RolesSecurityCtrl.prototype.createNewRole = function () {
        var _this = this;
        this.isAddNewRoleModalOpened = true;
        var modalInstance = this.$modal.open({
            templateUrl: 'ng1/dist/src/Administration/Security/Roles/CreateRoleNameModal/CreateRoleNameModal.html',
            windowClass: 'app-modal-window',
            controller: 'CreateRoleNameModalController',
            backdrop: 'static',
            resolve: {
                roles: function () {
                    return _this.roles;
                }
            }
        });
        modalInstance.result.then(function (name) {
            _this.isAddNewRoleModalOpened = false;
            _this.roleProvider.create(name).then(function (role) {
                var extendRole = _.deepclone(role);
                _this.notificationService.showMessage(String.Format(_this.language.getElementValue('securitySuccessCreateMsg'), name));
                _this.roleProvider.update(extendRole).then(function () {
                    _this.loadRoles(extendRole);
                }, function (error) {
                    _this.notificationService.errorMessage(_this.language.getElementValue('failedLoadDataServerMsg'), error);
                });
            }, function (error) {
                _this.notificationService.errorMessage(_this.language.getElementValue('failedLoadDataServerMsg'), error);
            });
        }, function () {
            _this.isAddNewRoleModalOpened = false;
        });
    };
    RolesSecurityCtrl.prototype.delete = function (role) {
        var _this = this;
        this.$modal
            .open({
            templateUrl: 'ng1/dist/src/Common/Components/YesNoModal.html',
            controller: 'yesNoModalCtrl',
            backdrop: 'static',
            resolve: {
                header: function () { return _this.language.getElementValue('securityRoleConfirmDeleteHdr'); },
                body: function () { return String.Format(_this.language.getElementValue('securityRoleConfirmDeleteMsg'), role.name); }
            }
        })
            .result.then(function () {
            _this.roleProvider.delete(role).then(function () {
                _this.loadRoles(_this.selectedRole);
                _this.authentication.updatePermissions();
                _this.notificationService.showMessage(String.Format(_this.language.getElementValue('securitySuccessDeleteMsg'), role.name));
            }, function (error) {
                _this.notificationService.errorMessage(_this.language.getElementValue('failedLoadDataServerMsg'), error);
            });
        }, function () { });
    };
    RolesSecurityCtrl.prototype.isSelectedRoleModified = function () {
        if (_.isUndefined(this.selectedRole)) {
            return false;
        }
        if (this.selectedRole.name !== this.originalRole.name) {
            return true;
        }
        if (this.selectedRole.permissions.length !== this.originalRole.permissions.length) {
            return true;
        }
        var originalRolePermissionsSorted = _.sortBy(this.originalRole.permissions, function (permission) { return permission.ident; });
        var newPermissionsSorted = _.sortBy(this.selectedRole.permissions, function (permission) { return permission.ident; });
        for (var i = 0; i < originalRolePermissionsSorted.length; i++) {
            if (originalRolePermissionsSorted[i].name !== newPermissionsSorted[i].name ||
                originalRolePermissionsSorted[i].ident !== newPermissionsSorted[i].ident) {
                return true;
            }
        }
        return false;
    };
    RolesSecurityCtrl.prototype.update = function () {
        var _this = this;
        if (_.any(this.roles, function (r) { return r.id !== _this.selectedRole.id && r.name === _this.selectedRole.name; })) {
            this.notificationService.warnMessage(String.Format(this.language.getElementValue('securityRoleExistsMsg'), this.selectedRole.name));
            return;
        }
        this.roleProvider.update(this.selectedRole).then(function () {
            _this.originalRole.name = _this.selectedRole.name;
            _this.originalRole.permissions = _this.selectedRole.permissions;
            _this.authentication.updatePermissions();
            _this.notificationService.showMessage(String.Format(_this.language.getElementValue('securitySuccessUpdateMsg'), _this.selectedRole.name));
        }, function (error) {
            return _this.notificationService.errorMessage(_this.language.getElementValue('failedLoadDataServerMsg'), error);
        });
    };
    RolesSecurityCtrl.prototype.changePermissionFilter = function () {
        this.tree.element.jstree().search(this.permissionFilter, true, true, false, false, true);
    };
    RolesSecurityCtrl.prototype.onCreateNewRole = function () {
        var _this = this;
        this.showSaveChangesModal(function () { return _this.createNewRole(); });
    };
    RolesSecurityCtrl.prototype.onChangeSelectedRole = function (role) {
        var _this = this;
        if (this.selectedRole !== role) {
            this.showSaveChangesModal(function () { return _this.select(role); });
        }
    };
    RolesSecurityCtrl.prototype.onDeleteRole = function (role) {
        this.delete(role);
    };
    RolesSecurityCtrl.inject = [
        '$scope',
        '$location',
        '$modal',
        'notificationService',
        'language',
        'roleProvider',
        'permissionProvider',
        'authentication'
    ];
    return RolesSecurityCtrl;
}());
angular.module('kamhpp').controller('RolesSecurityCtrl', RolesSecurityCtrl);
